<?php
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../models/Projects.php';

class User
{
    private $id;
    private $username;
    private $email;
    private $password;
    private $createdAt;

    // Getter pour l'ID utilisateur
    public function getId()
    {
        return $this->id;
    }
    // Getter pour l'email
    public function getEmail()
    {
        return $this->email;
    }

    public function getCreatedAt()
    {
        return $this->createdAt;
    }


    // Getter pour le nom d'utilisateur
    public function getUsername()
    {
        return $this->username;
    }

    // Inscription d'un utilisateur
    public function register($username, $email, $password)
    {
        $conn = connectDB();
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        try {
            $stmt = $conn->prepare("INSERT INTO users (nom, email, password, created_at) 
                                   VALUES (:nom, :email, :password, NOW())");
            $stmt->bindParam(':nom', $username, PDO::PARAM_STR);
            $stmt->bindParam(':email', $email, PDO::PARAM_STR);
            $stmt->bindParam(':password', $hashedPassword, PDO::PARAM_STR);
            $stmt->execute();
            $this->id = $conn->lastInsertId();
            $this->username = $username;
            $this->email = $email;

            return true;
        } catch (PDOException $e) {
            echo "Erreur : " . $e->getMessage();

            return false;
        }
    }

    // Connexion d'un utilisateur
    public function login($email, $password)
    {
        $conn = connectDB();

        try {
            $stmt = $conn->prepare("SELECT id, nom, email, password FROM users WHERE email = :email");
            $stmt->bindParam(':email', $email);
            $stmt->execute();

            if ($stmt->rowCount() === 1) {
                $user = $stmt->fetch(PDO::FETCH_ASSOC);

                if (password_verify($password, $user['password'])) {
                    $this->id = $user['id'];
                    $this->username = $user['nom'];
                    $this->email = $user['email'];

                    $_SESSION['user_id'] = $this->id;
                    $_SESSION['username'] = $this->username;

                    return true;
                }
            }
            return false;
        } catch (PDOException $e) {
            return false;
        }
    }

    // Récupérer un utilisateur par son ID
    public static function getById($id)
    {
        $conn = connectDB();

        try {
            $stmt = $conn->prepare("SELECT id, nom, email, created_at FROM users WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();

            if ($stmt->rowCount() === 1) {
                $userData = $stmt->fetch(PDO::FETCH_ASSOC);
                $user = new User();
                $user->id = $userData['id'];
                $user->username = $userData['nom'];
                $user->email = $userData['email'];
                $user->createdAt = $userData['created_at'];

                return $user;
            }
            return null;
        } catch (PDOException $e) {
            return null;
        }
    }

    // Récupérer les projets créés par l'utilisateur
    public function getProjects()
    {
        $conn = connectDB();

        try {
            $stmt = $conn->prepare("SELECT * FROM projects WHERE id_users = :id_users and statut != 'supprimé'");
            $stmt->bindParam(':id_users', $this->id);
            $stmt->execute();

            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            return [];
        }
    }

    // Récupérer les contributions de l'utilisateur
    public function getContributions()
    {
        $conn = connectDB();

        try {
            $stmt = $conn->prepare("SELECT c.*, p.titre as project_title 
                                   FROM contributions c 
                                   JOIN projects p ON c.id_projects = p.id 
                                   WHERE c.id_users = :id_users");
            $stmt->bindParam(':id_users', $this->id);
            $stmt->execute();

            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            return [];
        }
    }
    /**
     * recuperer le porteur d'un projet en fonction de l'id du projets
     * @param $id_projects l'identifiant du projet
     * @return mixed un tableau d'utilisateur
     */
    public function getUserByIdProject($id_projects)
    {
        $conn = connectDB();
        try {
            $stmt = $conn->prepare('SELECT *  FROM users u
                                    JOIN projects p ON u.id = p.id_users
                                    WHERE p.id = :id_projects');
            $stmt->bindParam(':id_projects', $id_projects);
            $stmt->execute();
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            return [];
        }
    }

    // Vérifier le mot de passe de l'utilisateur
    public function verifyPassword($userId, $password)
    {
        $conn = connectDB();

        try {
            $stmt = $conn->prepare("SELECT password FROM users WHERE id = :id");
            $stmt->bindParam(':id', $userId);
            $stmt->execute();

            if ($stmt->rowCount() === 1) {
                $user = $stmt->fetch(PDO::FETCH_ASSOC);
                return password_verify($password, $user['password']);
            }
            return false;
        } catch (PDOException $e) {
            return false;
        }
    }

    // Mettre à jour le profil de l'utilisateur
    public function updateProfile($userId, $username, $email, $newPassword = null)
    {
        $conn = connectDB();

        try {
            if ($newPassword !== null) {
                $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
                $stmt = $conn->prepare("UPDATE users SET nom = :nom, email = :email, password = :password WHERE id = :id");
                $stmt->bindParam(':password', $hashedPassword);
            } else {
                $stmt = $conn->prepare("UPDATE users SET nom = :nom, email = :email WHERE id = :id");
            }

            $stmt->bindParam(':nom', $username);
            $stmt->bindParam(':email', $email);
            $stmt->bindParam(':id', $userId);
            $stmt->execute();

            // Mettre à jour les propriétés de l'objet
            $this->id = $userId;
            $this->username = $username;
            $this->email = $email;

            return true;
        } catch (PDOException $e) {
            return false;
        }
    }
    // Déconnexion de l'utilisateur
    public static function logout()
    {
        session_start();
        session_unset();
        session_destroy();
        return true;
    }

    public function deleteAccount()
    {
        $conn = connectDB();
        try {
            // Supprimer d'abord les contributions de l'utilisateur
            $stmt = $conn->prepare("DELETE FROM contributions WHERE id_users = :id");
            $stmt->bindParam(':id', $this->id);
            $stmt->execute();

            // Supprimer ensuite les projets de l'utilisateur
            $stmt = $conn->prepare("DELETE FROM projects WHERE id_users = :id");
            $stmt->bindParam(':id', $this->id);
            $stmt->execute();

            // Enfin, supprimer l'utilisateur
            $stmt = $conn->prepare("DELETE FROM users WHERE id = :id");
            $stmt->bindParam(':id', $this->id);
            $stmt->execute();

            return true;
        } catch (PDOException $e) {
            return false;
        }
    }
    
    public function getUserByEmail($email)
    {
        $conn = connectDB();
        try {
            $stmt = $conn->prepare("SELECT * FROM users WHERE email=:email");
            $stmt->bindParam(':email', $email, PDO::PARAM_STR);
            $stmt->execute();
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($result) {
                return $result;
            } else {
                return [];
            }
        }catch(PDOException $e) {
            return [];
        }
    }
    public function updatePasswordByEmail(int $id, string $password)
    {
        $conn = connectDB();
        try {
            $stmt = $conn->prepare("UPDATE users SET password=:password WHERE id=:id ");
            $stmt->bindParam(":id", $id);
            $stmt->bindParam(":password", $password);
            $result = $stmt->execute();
            if ($result) {
                return true;
            }
            return false;
        }catch(PDOException $e) {
            return false;
        }
    }
}

// $u = new User();
// $c = $u->getContributions(1);
// print_r($c);
// 
?>